<?php

namespace App\Http\Controllers;

use App\Models\Work;
use App\Models\WorkType;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;

class WorkController extends Controller
{
    
    public function completedStatus(Request $request){
        $post = Work::find($request->id);
        $post->completed = $request->status;
        $post->save();
        return response()->json(['success'=>'Status successfully changed.']);
    }
    
    
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $items = Work::orderby('id','desc')->paginate(10);
        return view('back.work.index', compact('items'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $types = WorkType::orderBy('id','desc')->get();
        return view('back.work.create', compact('types'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|max:200',
            'work_type_id' => 'required',
            'description' => 'nullable|max:20000',
            'images.*' => 'image|mimes:jpg,jpeg,png,gif|max:2048',
        ]);

        $imagePaths = [];

        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $file) {
                $filename = uniqid() . '.' . $file->getClientOriginalExtension();
                $file->move(public_path('work/images'), $filename);
                $imagePaths[] = 'work/images/' . $filename;
            }
        }

        $item = new Work();
        $item->work_type_id = $request->work_type_id;
        $item->name = $request->name;
        $item->slug = Str::slug($request->name);
        $item->description = $request->description;
        $item->images = $imagePaths; // auto casted to JSON if column type is json
        $item->save();
        return Redirect()->back()->with('success','inserted successfully');
    }


    public function edit( $id)
    {
        $work = Work::find($id);
        $types = WorkType::orderBy('id','desc')->get();
        return view('back.work.edit', compact('types','work'));
    }

    public function update(Request $request, $id)
    {
        $work = Work::findOrFail($id);

        $request->validate([
            'work_type_id' => 'required|exists:work_types,id',
            'name' => 'required|max:200',
            'images.*' => 'image|mimes:jpg,jpeg,png,gif|max:2048',
        ]);

        $work->work_type_id = $request->work_type_id;
        $work->name = $request->name;
        $work->slug = Str::slug($request->name);
        $work->description = $request->description;

        // $imagePaths = $work->images ?? []; // আগের images

        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $file) {
                $filename = uniqid() . '.' . $file->getClientOriginalExtension();
                $file->move(public_path('uploads/images'), $filename);
                $imagePaths[] = 'uploads/images/' . $filename;
            }

        $work->images = $imagePaths;
        }

        $work->save();

        return redirect()->route('work.index')->with('success', 'Updated successfully');
    }
    
    
    public function destroy($id)
    {
        $work = Work::find($id);
    
        if (!$work) {
            return redirect()->back()->with('error', 'Work not found.');
        }
    
        
    
        // Delete images if exists
        $images = $work->images;
    
        // If it's a string, decode it; if it's already an array, leave it
        if (is_string($images)) {
            $images = json_decode($images, true);
        }
    
        if (is_array($images)) {
            foreach ($images as $image) {
                if (File::exists(public_path($image))) {
                    File::delete(public_path($image));
                }
            }
        }
        // Delete the work record
        $work->delete();
    
        return redirect()->back()->with('success', 'Work deleted successfully.');
    }
    

}
